/* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR REPRESENTATIONS,
 * EITHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING ANY IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY
 * OR COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. TI DISCLAIMS
 * ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET POSSESSION, AND NON-INFRINGEMENT
 * OF ANY THIRD PARTY INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
 * YOUR USE OF THE PROGRAM.
 * IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, CONSEQUENTIAL OR
 * INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY THEORY OF LIABILITY AND WHETHER OR
 * NOT TI HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY
 * OUT OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.  EXCLUDED
 * DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF REMOVAL OR REINSTALLATION,
 * COMPUTER TIME, LABOR COSTS, LOSS OF GOODWILL, LOSS OF PROFITS, LOSS OF
 * SAVINGS, OR LOSS OF USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
 * AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF YOUR USE OF THE
 * PROGRAM EXCEED FIVE HUNDRED DOLLARS (U.S.$500).
 * Unless otherwise stated, the Program written and copyrighted by Texas
 * Instruments is distributed as "freeware".  You may, only under TI's copyright
 * in the Program, use and modify the Program without any charge or restriction.
 * You may distribute to third parties, provided that you transfer a copy of this
 * license to the third party and the third party agrees to these terms by its
 * first use of the Program. You must reproduce the copyright notice and any
 * other legend of ownership on each copy or partial copy, of the Program.
 * You acknowledge and agree that the Program contains copyrighted material,
 * trade secrets and other TI proprietary information and is protected by
 * copyright laws, international copyright treaties, and trade secret laws, as
 * well as other intellectual property laws.  To protect TI's rights in the
 * Program, you agree not to decompile, reverse engineer, disassemble or
 * otherwise translate any object code versions of the Program to a
 * human-readable form.  You agree that in no event will you alter, remove or
 * destroy any copyright notice included in the Program.  TI reserves all rights
 * not specifically granted under this license. Except as specifically provided
 * herein, nothing in this agreement shall be construed as conferring by
 * implication, estoppel, or otherwise, upon you, any license or other right
 * under any TI patents, copyrights or trade secrets.
 * You may not use the Program in non-TI devices. */


//******************************************************************************
//
//  i2c.c
//  functions for using the I2C interface and ADS1015
//
//  Matthias Ulmann, Design Services - EMEA Power Management
//  Texas Instruments, Inc.
//
//******************************************************************************


#include "main.h"


static volatile unsigned int I2C_counter=0;
static volatile unsigned int I2C_send_configuration=0;
static volatile unsigned int I2C_select_conversion_register=0;
static volatile unsigned int I2C_read_conversion_register=0;
static volatile unsigned int I2C_MSB_AIN0, I2C_LSB_AIN0;
static volatile unsigned int I2C_MSB_AIN1, I2C_LSB_AIN1;



// This function manages the whole I2C communication.
// When the communication is disturbed, the mains and PFC voltage are set
// to 1000V.
void i2c_communication(void)
{
	// variable to check if the I2C bus is busy for a certain time or no ACK is received
	static volatile unsigned int I2C_busy=0;
	
	// If no slave is connected, UCNACKIFG is set and communication is started again.
	// But if one line is shorted to GND all time, BUSY flag is set and communication
	// stops in a loop.
	// If the BUSY flag is set for more than 5s, a fault is present and communication
	// is started again.
	I2C_busy++;
		
	if(I2C_busy > 50)	// I2C bus is for more than 5s busy -> fault!
	{
		I2C_busy = 0;				// reset flag
		
		clear_bit(IE2, UCB0TXIE);	// disable TX interrupt
		clear_bit(IE2, UCB0RXIE);	// disable RX interrupt
		
		clear_bit(IFG2, UCB0TXIFG);	// clear TX interrupt
		clear_bit(IFG2, UCB0RXIFG);	// clear RX interrupt
		
		// set voltages to 1000V to process the fault on the display
		voltage_mains = 1000;
		voltage_pfc = 1000;
		
		// initialize I2C registers again to reset UCBBUSY flag
		init_i2c();
		
		// reset modes
		I2C_send_configuration = 0;
		I2C_select_conversion_register = 0;	
		I2C_read_conversion_register = 0;
		
		I2C_flow = 0;				// restart communication from beginning
	}
	
	
	// 0: send configuration for ADCIN0
	// 1: select conversion register (ADCIN0)
	// 2: read conversion register of ADCIN0
	// 3: send configuration for ADCIN1
	// 4: select conversion register (ADCIN1)
	// 5: read conversion register of ADCIN1
	// If a problem occurs the flow starts at 0 again.
	
	// Check if previous communication was finished otherwise skip.
	// UCBOTXIE/RXIE are cleared when the communication (RX/TX)
	// is finished.
	if( !(IE2 & UCB0TXIE) & !(IE2 & UCB0RXIE) )
	{
		switch(I2C_flow)
		{
			case 0:	I2C_channel = 0;	// set channel for ISR
					ic2_send_configuration(I2C_channel);
					I2C_flow++;
					I2C_busy=0;
					break;
		
			case 1: ic2_select_conversion_register();
					I2C_flow++;
					I2C_busy=0;
					break;
					
			case 2: i2c_read_conversion_register();
					I2C_flow++;
					I2C_busy=0;
					break;
					
			case 3:	I2C_channel = 1;	// set channel for ISR
					ic2_send_configuration(I2C_channel);
					I2C_flow++;
					I2C_busy=0;
					break;
			
			case 4: ic2_select_conversion_register();
					I2C_flow++;
					I2C_busy=0;
					break;
					
			case 5: i2c_read_conversion_register();
					I2C_flow = 0;
					I2C_busy=0;
					break;
		}
	}
}



// Conversion of the sampled digital value into voltage.
// The value has three digits without decimal place.
unsigned int lsb_to_voltage_i2c(unsigned int lsb)
{
	unsigned long voltage_long;
	unsigned int voltage_int;
	
	// The input voltage is attenuated by a voltage divider 3M/23k2.
	// For the mains voltage, the voltage has also to be multiplied by
	// PI/(2* SQRT(2)) to calculate the RMS voltage.
	// This formula is used to calculate the mains voltage from the LSBs:
	// Vmains = LSB * (4.096/2047) * (23k2 + 3M)/23k2 * PI/(2* SQRT(2))
	// simplified: Vmains = LSB * 0.290
	// For the PFC voltage only the voltage divider has to be considered.
	// Vpfc = LSB * (4.096/2047) * (23k2 + 3M)/23k2
	// simplified: Vpfc = LSB * 0.261
	// To prevent an overflow and the loss of the first decimal place a
	// long integer is needed for calculation.
	
	if(I2C_channel == 0)	// calculate PFC voltage
	{
		voltage_long = (unsigned long)lsb * 261;
		voltage_long = voltage_long / 1000;
		
		voltage_int = voltage_long;
	}
	
	if(I2C_channel == 1)	// calculate mains voltage
	{
		voltage_long = (unsigned long)lsb * 290;
		voltage_long = voltage_long / 1000;
		
		// From 0 to 0.5A output current the measurement of the 
		// RMS mains voltage shows a too high value. Thus a correction
		// factor is applied.
		// Real voltage: 214V, calculated voltage: 300V
		if(current_output <= 5)
			{
			// The calculated voltage is divided by 1.4
			voltage_long = voltage_long * 5;
			voltage_long = voltage_long / 7;
			}
		
		voltage_int = voltage_long;
	}
	return voltage_int;
}



// Initialization of the I2C interface
void init_i2c(void)
{
	// enable software reset
	set_bit(UCB0CTL1, UCSWRST);
	
	// set own address to 7 bit
	clear_bit(UCB0CTL0, UCA10);
	
	// set slave address to 7 bit
	clear_bit(UCB0CTL0, UCSLA10);
	
	// no multi-master environment
	clear_bit(UCB0CTL0, UCMM);
	
	// set master mode
	set_bit(UCB0CTL0, UCMST);
	
	// set I2C mode
	set_bit(UCB0CTL0, UCMODE1);
	set_bit(UCB0CTL0, UCMODE0);
	
	// set synchronous mode
	set_bit(UCB0CTL0, UCSYNC);
	
	// select SMCLK as clock source
	set_bit(UCB0CTL1, UCSSEL1);
	set_bit(UCB0CTL1, UCSSEL0);
	
	// normal acknowledge
	clear_bit(UCB0CTL1, UCTXNACK);
	
	// set bus frequency
	// 16 MHz / 2000 = 8 kHz
	//  -> UC0BR1 = 0x07, UCB0BR0 = 0xD0, custom ultra-slow mode
	// 16 MHz / 160 = 100 kHz
	//  -> UC0BR1 = 0x00, UCB0BR0 = 0xA0, standard
	// 16 MHz / 40 = 400 kHz
	//  -> UC0BR1 = 0x00, UCB0BR0 = 0x28, fast mode
	// 16 MHz / 16 = 1000 kHz
	//  -> UC0BR1 = 0x00, UCB0BR0 = 0x05, fast mode plus
	UCB0BR1 = 0x07;
	UCB0BR0 = 0xD0;
	
	// set slave address
	UCB0I2CSA = SLAVE_ADDRESS;
	
	// set MSP430 as receiver
	clear_bit(UCB0CTL1, UCTR);
	
	// choose I2C function for pins
	set_bit(SCL_PSEL, SCL_BIT);
	set_bit(SDA_PSEL, SDA_BIT);
	
	// enable interrupt when ACK is expected but not received
	set_bit(UCB0I2CIE, UCNACKIE);
	
	// enable interrupt when arbitration is lost
	set_bit(UCB0I2CIE, UCALIE);
	
	// disable software reset
	clear_bit(UCB0CTL1, UCSWRST);
}



// Send configuration to ADS1015
// 0: PFC, 1: mains
void ic2_send_configuration(unsigned int channel)
{
	// set configuration mode
   	I2C_send_configuration = 1;
   	
   	// set channel
   	if(channel == 0)
   	{
   		I2C_channel = 0;
   	}
   	if(channel == 1)
   	{
   		I2C_channel = 1;
   	}
   	
   	// set counter for configuration mode
   	I2C_counter = 3;
			
	// set MSP430 as transmitter
	set_bit(UCB0CTL1, UCTR);
	
	// enable TX interrupt
	set_bit(IE2, UCB0TXIE);
			
	// send START condition and slave address
	set_bit(UCB0CTL1, UCTXSTT);	
}



// Select conversion register of ADS1015 for reading
void ic2_select_conversion_register(void)
{
	// select conversion register
   	I2C_select_conversion_register = 1;
   	
   	// set counter
   	I2C_counter = 1;
			
	// set MSP430 as transmitter
	set_bit(UCB0CTL1, UCTR);
	
	// enable TX interrupt
	set_bit(IE2, UCB0TXIE);
			
	// send START condition and slave address
	set_bit(UCB0CTL1, UCTXSTT);	
}



// Read conversion register of ADS1015
void i2c_read_conversion_register(void)
{
	// set read mode
   	I2C_read_conversion_register = 1;
   	
   	// set counter
   	I2C_counter = 1;
			
	// set MSP430 as receiver
	clear_bit(UCB0CTL1, UCTR);
	
	// enable RX interrupt
	set_bit(IE2, UCB0RXIE);
			
	// send START condition and slave address
	set_bit(UCB0CTL1, UCTXSTT);
}



// Executed when the I2C TX or RX interrupt flag is set. This occurs when the
// device is in TX mode and the buffer is empty or when it is in RX mode and
// the buffer is full.
#pragma vector = USCIAB0TX_VECTOR
__interrupt void USCIAB0TX_ISR(void)
{	
	// static variables for filtering
	static volatile unsigned int voltage_pfc_LSB_old=0, voltage_pfc_LSB_new=0;
	
	// static variables for filtering
	static volatile unsigned int voltage_mains_LSB_old=0, voltage_mains_LSB_new=0;
	
	unsigned int calculation_lsb;
	
	
	// Filter constant (Linear Shift-Invariant)
	const unsigned int analog_averaging = 8;
	
	
	if(I2C_send_configuration)
	{
		switch(I2C_counter)
		{
			case 3:	UCB0TXBUF = 0x01;		// select configuration register
					break;
			
			case 2: if(I2C_channel == 0)
					{
						// select channel AIN0, PGA gain=1, continuous conversion mode
						UCB0TXBUF = 0x42;
					}
					if(I2C_channel == 1)
					{
						// select channel AIN1, PGA gain=1, continuous conversion mode
						UCB0TXBUF = 0x52;
					}	
					break;
					
			case 1:	// select 128 SPS, comparator off
					UCB0TXBUF = 0x03;
					break;
					
			case 0: set_bit(UCB0CTL1, UCTXSTP);		// send STOP condition
					clear_bit(IE2, UCB0TXIE);		// disable TX interrupt
					I2C_send_configuration = 0;
					break;
		}
	}
	
	
	if(I2C_select_conversion_register)
	{
		switch(I2C_counter)
		{
			case 1:	UCB0TXBUF = 0x00;	// select conversion register
					break;
					
			case 0: set_bit(UCB0CTL1, UCTXSTP);		// send STOP condition
					clear_bit(IE2, UCB0TXIE);		// disable TX interrupt
					I2C_select_conversion_register = 0;
					break;
		}
	}
	
	
	if(I2C_read_conversion_register)
	{
		switch(I2C_counter)
		{
			case 1: if(I2C_channel == 0)
					{
						I2C_MSB_AIN0 = UCB0RXBUF;		// read RX buffer
						set_bit(UCB0CTL1, UCTXSTP);		// send STOP condition
					}
					if(I2C_channel == 1)
					{
						I2C_MSB_AIN1 = UCB0RXBUF;		// read RX buffer
						set_bit(UCB0CTL1, UCTXSTP);		// send STOP condition
					}
					break;
					
			case 0:	if(I2C_channel == 0)	// PFC
					{
						I2C_LSB_AIN0 = UCB0RXBUF;		// read RX buffer
						clear_bit(IE2, UCB0RXIE);		// disable RX interrupt
						
						voltage_pfc_LSB_old = voltage_pfc_LSB_new;	// save last value
						
						voltage_pfc_LSB_new = 0;									// set all bits to zero
						voltage_pfc_LSB_new = voltage_pfc_LSB_new ^ I2C_MSB_AIN0;	// XOR with MSB
						voltage_pfc_LSB_new = voltage_pfc_LSB_new<<8;				// shift 8 left
						voltage_pfc_LSB_new = voltage_pfc_LSB_new ^ I2C_LSB_AIN0;	// XOR with LSB
						voltage_pfc_LSB_new = voltage_pfc_LSB_new>>4;				// shift 4 right
						
						// filter new value
						calculation_lsb =
						voltage_pfc_LSB_old * (analog_averaging - 1)/analog_averaging
						+ (voltage_pfc_LSB_new / analog_averaging);
				
						// calculated input voltage
						voltage_pfc = lsb_to_voltage_i2c(calculation_lsb);
					}
					
					if(I2C_channel == 1)	// mains
					{
						I2C_LSB_AIN1 = UCB0RXBUF;		// read RX buffer
						clear_bit(IE2, UCB0RXIE);		// disable RX interrupt
						
						voltage_mains_LSB_old = voltage_mains_LSB_new;	// save last value
						
						voltage_mains_LSB_new = 0;										// set all bits to zero
						voltage_mains_LSB_new = voltage_mains_LSB_new ^ I2C_MSB_AIN1;	// XOR with MSB
						voltage_mains_LSB_new = voltage_mains_LSB_new<<8;				// shift 8 left
						voltage_mains_LSB_new = voltage_mains_LSB_new ^ I2C_LSB_AIN1;	// XOR with LSB
						voltage_mains_LSB_new = voltage_mains_LSB_new>>4;				// shift 4 right
						
						// filter new value
						calculation_lsb =
						voltage_mains_LSB_old * (analog_averaging - 1)/analog_averaging
						+ (voltage_mains_LSB_new / analog_averaging);
						
						// calculated input voltage
						voltage_mains = lsb_to_voltage_i2c(calculation_lsb);
					}
					
					I2C_read_conversion_register = 0;
					break;
		}
	}
	

	I2C_counter--;
		
	clear_bit(IFG2, UCB0TXIFG);		// clear TX interrupt
	clear_bit(IFG2, UCB0RXIFG);		// clear RX interrupt
}



// Executed when the I2C interrupt flags for "Arbitration-lost" or
// "Not-acknowledge" is set.
#pragma vector = USCIAB0RX_VECTOR
__interrupt void USCIAB0RX_ISR(void)
{
	// start I2C communication from beginning
	I2C_flow=0;
	
	// send STOP
	set_bit(UCB0CTL1, UCTXSTP);		// send STOP condition
	clear_bit(IE2, UCB0TXIE);		// disable TX interrupt
	clear_bit(IE2, UCB0RXIE);		// disable RX interrupt
	
	// reset modes
	I2C_send_configuration = 0;
	I2C_select_conversion_register = 0;	
	I2C_read_conversion_register = 0;
	
	// set voltages to 1000V to process the fault on the display
	voltage_mains = 1000;
	voltage_pfc = 1000;
	
	clear_bit(IFG2, UCB0TXIFG);		// clear TX interrupt
	clear_bit(IFG2, UCB0RXIFG);		// clear RX interrupt
	
	clear_bit(UCB0STAT, UCNACKIFG);		// clear NACK flag
	clear_bit(UCB0STAT, UCALIFG);		// clear arbitration-lost flag
}
